//*****************************************************************
//* MSX PPR/STG (VScreen Editor) converter
//*****************************************************************

#include <stdio.h>
#include <stdlib.h>

//#define	EXACTCOLOR	0

typedef	unsigned char	BYTE;
typedef	unsigned short	WORD;

// Load default palette
void loadDefaultPalette();

// Load SC4 file
void loadSC4(char* _szFilename);

// Load PPR file
void loadPPR(char* _szFilename);

// Write files
void writeFiles(char* _szPrefix);

// Remap color table
void remapColorTable();

// Find color in default palette
int findColor(WORD _uColor);

// Find nearest color in default palette. Only used if EXACTCOLOR is not defined
int findNearestColor(WORD _uColor);

// Default palette
WORD*	pDefaultPalette	= NULL;

// Palette
WORD*	pPalette		= NULL;

// Pattern table
BYTE*	pPatternTable	= NULL;

// Color table
BYTE*	pColorTable		= NULL;

void Usage();

int main(int argc, char *argv[])
{
	printf("MSX PPR (VScreen Editor) converter.\n");

	if (argc != 3)
	{
		Usage();
	}

	// Load the TMS9918 palette.
	loadDefaultPalette();
	
	// Load the PPR file
	loadPPR(argv[1]);
	
	// Remap color table to match the TMS9918 palette.
	remapColorTable();
	
	// Write files
	writeFiles(argv[2]);
	
	if (pDefaultPalette != NULL)
	{
		delete [] pDefaultPalette;
		
		pDefaultPalette	= NULL;
	}

	if (pPatternTable != NULL)
	{
		delete [] pPatternTable;
		
		pPatternTable	= NULL;
	}

	if (pColorTable != NULL)
	{
		delete [] pColorTable;
		
		pColorTable	= NULL;
	}

	if (pPalette != NULL)
	{
		delete [] pPalette;
		
		pPalette	= NULL;
	}
}

void Usage()
{
	printf("Command syntax is: ScreenMap Infile.ppr OutPrefix\n");

	exit(1);
}

void loadDefaultPalette()
{
	pDefaultPalette	= new WORD[16];
	
	FILE*	pHandle	= fopen("Default.pal", "rb");
	
	if (NULL == pHandle)
	{
		printf("Unable to load default palette file Default.pal\n");
	
		exit(1);
	}
	
	fseek(pHandle, 0, SEEK_END);
	
	if (ftell(pHandle) != 32)
	{
		printf("Default palette file (Default.pal) is corrupt\n");
	
		exit(1);
	}
	
	rewind(pHandle);
	
	fread(pDefaultPalette, 2, 16, pHandle);
	
	fclose(pHandle);
}

void loadSC4(char* _szFilename)
{
	pPatternTable	= new BYTE[2048];
	pColorTable		= new BYTE[2048];
	pPalette		= new WORD[16];

	FILE*	pHandle	= fopen(_szFilename, "rb");
	
	if (NULL == pHandle)
	{
		printf("Unable to load SC4 file %s\n", _szFilename);
	
		exit(1);
	}
	
	fseek(pHandle, 0, SEEK_END);
	
	if (ftell(pHandle) != 32 + 2048 + 2048)
	{
		printf("SC4 file (%s) is corrupt\n", _szFilename);
	
		exit(1);
	}
	
	rewind(pHandle);
	
	fread(pPalette, 2, 16, pHandle);
	fread(pPatternTable, 1, 2048, pHandle);
	fread(pColorTable, 1, 2048, pHandle);
	
	fclose(pHandle);
}

void loadPPR(char* _szFilename)
{
	pPatternTable	= new BYTE[2048];
	pColorTable		= new BYTE[2048];
	pPalette		= new WORD[16];

	FILE*	pHandle	= fopen(_szFilename, "rb");
	
	if (NULL == pHandle)
	{
		printf("Unable to load PPR file %s\n", _szFilename);
	
		exit(1);
	}
	
	fseek(pHandle, 0, SEEK_END);
	
	if (ftell(pHandle) != 4392)
	{
		printf("PPR file (%s) is corrupt\n", _szFilename);
	
		exit(1);
	}
	
	rewind(pHandle);
	
	fseek(pHandle, 7, SEEK_CUR);
	
	fread(pPatternTable, 1, 2048, pHandle);

	fread(pColorTable, 1, 2048, pHandle);

	fseek(pHandle, 1, SEEK_CUR);

	fread(pPalette, 2, 16, pHandle);
	
	fclose(pHandle);

	for (int iLoop = 0; iLoop < 2048; ++iLoop)
	{
		BYTE	uValue1	= pColorTable[iLoop] & 0x0F;
		BYTE	uValue2	= pColorTable[iLoop] & 0xF0;
		
		pColorTable[iLoop]	= (uValue1 << 4) | (uValue2 >> 4);
	}

	for (int iLoop = 0; iLoop < 2048; ++iLoop)
	{
		int	iValue	= pPatternTable[iLoop];		
		
		pPatternTable[iLoop]	= (BYTE)(((iValue * 0x0802LU & 0x22110LU) | (iValue * 0x8020LU & 0x88440LU)) * 0x10101LU >> 16);
	}
}

void writeFiles(char* _szPrefix)
{
	char	szFilename[256];
	
	sprintf(szFilename, "%s.pat", _szPrefix);
	
	FILE*	pHandle	= fopen(szFilename, "wb");
	
	if (NULL == pHandle)
	{
		printf("Unable to create pattern file %s\n", szFilename);
	
		exit(1);
	}

	fwrite(pPatternTable, 1, 2048, pHandle);
	
	fclose(pHandle);

	sprintf(szFilename, "%s.col", _szPrefix);
	
	pHandle	= fopen(szFilename, "wb");
	
	if (NULL == pHandle)
	{
		printf("Unable to create color file %s\n", szFilename);
	
		exit(1);
	}

	fwrite(pColorTable, 1, 2048, pHandle);
	
	fclose(pHandle);
}

void remapColorTable()
{
	int*	pPaletteIndexes	= new int[16];
	
	pPaletteIndexes[0]	= 0;
	
	for (int iLoop = 1; iLoop < 16; ++iLoop)
	{
#ifdef EXACTCOLOR
		int	iIndex	= findColor(pPalette[iLoop]);

		if (-1 == iIndex)
		{
			printf("Palette index %d does not match fixed palette\n", iLoop);
		
			exit(1);
		}
#else
		int	iIndex	= findNearestColor(pPalette[iLoop]);
#endif
	
		pPaletteIndexes[iLoop]	= iIndex;
	}

	for (int iLoop = 0; iLoop < 2048; ++iLoop)
	{
		int	iBackgroundPalette	= pPaletteIndexes[pColorTable[iLoop] & 0x0F];
		int	iForegroundPalette	= pPaletteIndexes[(pColorTable[iLoop] & 0xF0) >> 4];
	
		pColorTable[iLoop]	= (iForegroundPalette << 4) | iBackgroundPalette;
	}
	
	delete [] pPaletteIndexes;
}

int findColor(WORD _uColor)
{
	for (int iLoop = 1; iLoop < 16; ++iLoop)
	{
		if (pDefaultPalette[iLoop] == _uColor)
		{
			return	iLoop;
		}
	}

	return	-1;
}

int findNearestColor(WORD _uColor)
{
	int	iColor	= -1;
	int	iError	= 100000000;
	
	BYTE	uRed	= (_uColor & 0xF00) >> 8;
	BYTE	uGreen	= (_uColor & 0x0F0) >> 4;
	BYTE	uBlue	= (_uColor & 0x00F);
	
	for (int iLoop = 1; iLoop < 16; ++iLoop)
	{
		WORD	uNearestColor	= pDefaultPalette[iLoop];
		
		BYTE	uNearestRed		= (uNearestColor & 0xF00) >> 8;
		BYTE	uNearestGreen	= (uNearestColor & 0x0F0) >> 4;
		BYTE	uNearestBlue	= (uNearestColor & 0x00F);
		
		int	iMatch	= (1 << abs(uNearestRed - uRed)) + (1 << abs(uNearestGreen - uGreen)) + (1 << abs(uNearestBlue - uBlue));
		
		if (iMatch < iError)
		{
			iError	= iMatch;
			iColor	= iLoop;
		}
	}

	return	iColor;
}